/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:    All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.    The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true*/
/*global describe: true, _: true, expect: true, window: true, WebSocket: true, it: true, afterEach */
"use strict";

var ConcurrencyLimiter = require("../lib/asset-exporter/concurrency-limiter"),
    Q = require("q"),
    os = require("os"),
    sinon = require("sinon");

describe('ConcurrencyLimiter', function() {
    var sandbox = sinon.sandbox.create();

    afterEach(function () {
        sandbox.restore();
    });

    function startNewTest(maxJobs, totalJobs) {
        var test = {
            maxJobs: maxJobs,
            totalJobs: totalJobs,
            numJobsStarted: 0,
            jobs: [],
        };

        test.concurrencyLimiter = new ConcurrencyLimiter(test.maxJobs);

        for (var i = 0; i < test.totalJobs; i++) {
            // Create a job and a function to kick it off.
            var job = Q.defer();
            test.jobs[i] = job;

            var promiseFactory = function(job) {
                test.numJobsStarted++;
                return job.promise;
            }.bind(null, job);

            // Enqueue the job.
            test.concurrencyLimiter.enqueue(promiseFactory);
        }

        return test;
    }

    function fulfillAll(promises, rejectOrResolve) {
        promises.map(function(promises) {
            promises[rejectOrResolve]();
        });
        return Q.allSettled(promises);
    }

    it("should limit the number of jobs running in parallel", function() {
        var test = startNewTest(4, 8);

        // Expect the maximum number of jobs to have been started.
        expect(test.numJobsStarted).to.equal(test.maxJobs);
    });

    it("should limit the number of jobs to the number of cores by default", function() {
        // Make the OS return 3 fake cores when asked.
        var fakeCores = [{}, {}, {}];
        sandbox.stub(os, "cpus").returns(fakeCores);

        // Pass null for maxJobs, so we use the number of cores by default.
        var test = startNewTest(null, 8);

        // Expect the number of started jobs to be limited by the number of cores.
        expect(test.numJobsStarted).to.equal(fakeCores.length);
    });

    it("should start new jobs when others finish successfully", function() {
        var test = startNewTest(4, 8);

        var jobsToFinish = test.jobs.slice(0, 2);
        var promise = fulfillAll(jobsToFinish, "resolve");
        return expect(promise).to.eventually.be.fulfilled.then(function() {
            // The number of additional jobs started should equal the number of jobs finished.
            expect(test.numJobsStarted).to.equal(test.maxJobs + jobsToFinish.length);
        });
    });

    it("should start new jobs when others fail", function() {
        var test = startNewTest(4, 8);

        var jobsToFinish = test.jobs.slice(0, 2);
        var promise = fulfillAll(jobsToFinish, "reject");
        return expect(promise).to.eventually.be.fulfilled.then(function() {
            // The number of additional jobs started should equal the number of jobs finished.
            expect(test.numJobsStarted).to.equal(test.maxJobs + jobsToFinish.length);
        });
    });

    it("should start all queued jobs eventually", function() {
        var test = startNewTest(4, 8);

        var jobsToFinish = test.jobs.slice(0, test.totalJobs);
        var promise = fulfillAll(jobsToFinish, "resolve");
        return expect(promise).to.eventually.be.fulfilled.then(function() {
            expect(test.numJobsStarted).to.equal(test.totalJobs);
        });
    });
});
